
********************************************************************************
*                                                                              *
* 1 by 4 within subject design: Example 8                                      *
*                                                                              *
********************************************************************************

clear
set more off

use within1by4data
browse

**********************
* Conventional ANOVA *
**********************

// Again, we start with conducting a conventional ANOVA to obtain SSmodel. To 
// conduct a conventional within-subject ANOVA, Stata requires the data to be 
// in long format. Therefore the dataset needs to be reshaped in to long format:

reshape long row, i(person) j(location)  
rename row retention
browse

// Next we run the rconventional within-subject ANOVA with retention as the 
// dependent variable, and location as factor:

anova retention location person, repeated(location) 

// In contrast to SPSS, Stata includes in the ANOVA output the SSmodel. However,
// this SSmodel is defined differently than in the manuscript, and includes both
// the SSlocation and SSperson. Also the SSlocation*person in Equation 8 is 
// labeled SSresidual in Stata output.  

// To obtain the SSmodel as defined in the manuscript, we need thus to subtract
// from the SSmodel in Stata output (stored in Stata's memory as "e(mss)" the 
// the SSperson (stored in Stata's memory as "e(SS_2)" (SS_2 because person is 
// the second factor listed in the ANOVA procedure above. Let us store 
// the required SSmodel (i.e., the observed variance due to the location 
// manipulation) in Mata's memory: 

mata
SSmodel = st_numscalar("e(mss)") - st_numscalar("e(ss_2)")
SSmodel
end

// The SSmodel equals SSlocation in this single factor repeated measures design,
// because only seating location was manipulated in this example dataset.

***************************************
* Example 8: A negative linear trend? *
***************************************

// As in the between-subject case, constrast analysis on repeated measures 
// designs in STATA  uses the manovatest postestimation command of the MANOVA 
// procedure. However, in  contrast to between-subject designs, the set of 
// contrast weights are to be specified in the so-called transformation matrix 
// M, which is input to the ytransform option of manovatest.

// Since MANOVA requires data to be in wide format, let us re-open the original
// dataset:

use within1by4data, clear
browse

// The example involves data from a fully within-subject design without 
// between-subject factor. The MANOVA procedure, however, requires a 
// between-subject grouping variable. Therefore, we first create a variable 
// called "groups" which classifies all participants as belonging to the same 
// group:

gen groups = 1

// Next we conduct the MANOVA with the four repeated retention measurements 
// (variables row1 to row4) as dependent variables, and the "groups" variable as
// factor. Since there is only a single between-subject group no intercept or 
// constant can be defined, so the noconstant option is used. As in previous 
// examples, the output of the MANOVA itself is of little interest:

manova row1 row2 row3 row4 = groups, noconstant

// Next, we define the row vector M that contains the contrast weights for each
// cell in the design. No intercept weight is included in this vector M:

matrix M = (3, 1, -1, -3)

// Then, we use the manovatest subestimation command to conduct the contrast 
// hypothesis test. In contrast to between-subject designs (Examples 1 to 7),
// the between-subject grouping variable (groups) needs to be specified, and
// the M matrix is input to the ytransform option:

manovatest groups, ytransform(M)

// The SScontrast is, as before, saved in the matrix r(H), which contrains a
// single value. Let us obtain this and store it Mata's memory so that it is
// available when calculating ETA^2 alerting later on:

mata
SScontrast = st_matrix("r(H)")
end

// Next let us first calculate the C-estimate, which requires the cell means:

means row1 row2 row3 row4

// Next, we can calculate the C-estimate in the same way as in previous examples
// by calculating the weighted sum of cell means using the weights specified in 
// M:

mata
Means = st_matrix("e(b)")
M = st_matrix("M")
Cest = rowsum(M :* Means) 
Cest
end

// Finally we calculate ETA^alerting using the SScontrast stored in Mata's
// memory above. Since an M matrix is involved the SScontrast has first to be
// divided by the sum of squared M matrix weights, before it can be compared to
// the SSmodel obtained from the conventional ANOVA:

mata
SScontrast_adj = SScontrast / rowsum(M:*M)
SScontrast_adj
Eta2alerting = SScontrast_adj / SSmodel
Eta2alerting
end



